// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#include "exi_bba.h"
#include "tap-win32/common.h"
#include "main.h"


#if 0
/*
* Confirm that GUID is a TAP-Win32 adapter.
*/
static bool
is_tap_win32 (const char *guid, const struct tap_reg *tap_reg)
{
	const struct tap_reg *tr;

	for (tr = tap_reg; tr != NULL; tr = tr->next)
	{
		if (guid && !strcmp (tr->guid, guid))
			return true;
	}

	return false;
}

static const char *
guid_to_name (const char *guid, const struct panel_reg *panel_reg)
{
	const struct panel_reg *pr;

	for (pr = panel_reg; pr != NULL; pr = pr->next)
	{
		if (guid && !strcmp (pr->guid, guid))
			return pr->name;
	}

	return NULL;
}

static const char *
name_to_guid (const char *name, const struct tap_reg *tap_reg, const struct panel_reg *panel_reg)
{
	const struct panel_reg *pr;

	for (pr = panel_reg; pr != NULL; pr = pr->next)
	{
		if (name && !strcmp (pr->name, name) && is_tap_win32 (pr->guid, tap_reg))
			return pr->guid;
	}

	return NULL;
}

const struct tap_reg *
get_tap_reg (struct gc_arena *gc)
{
	HKEY adapter_key;
	LONG status;
	DWORD len;
	struct tap_reg *first = NULL;
	struct tap_reg *last = NULL;
	int i = 0;

	status = RegOpenKeyEx(
		HKEY_LOCAL_MACHINE,
		ADAPTER_KEY,
		0,
		KEY_READ,
		&adapter_key);

	if (status != ERROR_SUCCESS)
		DEGUB("Error opening registry key: %s", ADAPTER_KEY);

	while (true)
	{
		char enum_name[256];
		char unit_string[256];
		HKEY unit_key;
		char component_id_string[] = "ComponentId";
		char component_id[256];
		char net_cfg_instance_id_string[] = "NetCfgInstanceId";
		char net_cfg_instance_id[256];
		DWORD data_type;

		len = sizeof (enum_name);
		status = RegEnumKeyEx(
			adapter_key,
			i,
			enum_name,
			&len,
			NULL,
			NULL,
			NULL,
			NULL);
		if (status == ERROR_NO_MORE_ITEMS)
			break;
		else if (status != ERROR_SUCCESS)
			DEGUB("Error enumerating registry subkeys of key: %s",
			ADAPTER_KEY);

		openvpn_snprintf (unit_string, sizeof(unit_string), "%s\\%s",
			ADAPTER_KEY, enum_name);

		status = RegOpenKeyEx(
			HKEY_LOCAL_MACHINE,
			unit_string,
			0,
			KEY_READ,
			&unit_key);

		if (status != ERROR_SUCCESS)
			dmsg (D_REGISTRY, "Error opening registry key: %s", unit_string);
		else
		{
			len = sizeof (component_id);
			status = RegQueryValueEx(
				unit_key,
				component_id_string,
				NULL,
				&data_type,
				component_id,
				&len);

			if (status != ERROR_SUCCESS || data_type != REG_SZ)
				dmsg (D_REGISTRY, "Error opening registry key: %s\\%s",
				unit_string, component_id_string);
			else
			{	      
				len = sizeof (net_cfg_instance_id);
				status = RegQueryValueEx(
					unit_key,
					net_cfg_instance_id_string,
					NULL,
					&data_type,
					net_cfg_instance_id,
					&len);

				if (status == ERROR_SUCCESS && data_type == REG_SZ)
				{
					if (!strcmp (component_id, TAP_COMPONENT_ID))
					{
						struct tap_reg *reg;
						ALLOC_OBJ_CLEAR_GC (reg, struct tap_reg, gc);
						reg->guid = string_alloc (net_cfg_instance_id, gc);

						/* link into return list */
						if (!first)
							first = reg;
						if (last)
							last->next = reg;
						last = reg;
					}
				}
			}
			RegCloseKey (unit_key);
		}
		++i;
	}

	RegCloseKey (adapter_key);
	return first;
}

const struct panel_reg *
get_panel_reg (struct gc_arena *gc)
{
	LONG status;
	HKEY network_connections_key;
	DWORD len;
	struct panel_reg *first = NULL;
	struct panel_reg *last = NULL;
	int i = 0;

	status = RegOpenKeyEx(
		HKEY_LOCAL_MACHINE,
		NETWORK_CONNECTIONS_KEY,
		0,
		KEY_READ,
		&network_connections_key);

	if (status != ERROR_SUCCESS)
		DEGUB("Error opening registry key: %s", NETWORK_CONNECTIONS_KEY);

	while (true)
	{
		char enum_name[256];
		char connection_string[256];
		HKEY connection_key;
		char name_data[256];
		DWORD name_type;
		const char name_string[] = "Name";

		len = sizeof (enum_name);
		status = RegEnumKeyEx(
			network_connections_key,
			i,
			enum_name,
			&len,
			NULL,
			NULL,
			NULL,
			NULL);
		if (status == ERROR_NO_MORE_ITEMS)
			break;
		else if (status != ERROR_SUCCESS)
			DEGUB("Error enumerating registry subkeys of key: %s",
			NETWORK_CONNECTIONS_KEY);

		openvpn_snprintf (connection_string, sizeof(connection_string),
			"%s\\%s\\Connection",
			NETWORK_CONNECTIONS_KEY, enum_name);

		status = RegOpenKeyEx(
			HKEY_LOCAL_MACHINE,
			connection_string,
			0,
			KEY_READ,
			&connection_key);

		if (status != ERROR_SUCCESS)
			dmsg (D_REGISTRY, "Error opening registry key: %s", connection_string);
		else
		{
			len = sizeof (name_data);
			status = RegQueryValueEx(
				connection_key,
				name_string,
				NULL,
				&name_type,
				name_data,
				&len);

			if (status != ERROR_SUCCESS || name_type != REG_SZ)
				dmsg (D_REGISTRY, "Error opening registry key: %s\\%s\\%s",
				NETWORK_CONNECTIONS_KEY, connection_string, name_string);
			else
			{
				struct panel_reg *reg;

				ALLOC_OBJ_CLEAR_GC (reg, struct panel_reg, gc);
				reg->name = string_alloc (name_data, gc);
				reg->guid = string_alloc (enum_name, gc);

				/* link into return list */
				if (!first)
					first = reg;
				if (last)
					last->next = reg;
				last = reg;
			}
			RegCloseKey (connection_key);
		}
		++i;
	}

	RegCloseKey (network_connections_key);

	return first;
}

/*
* Lookup a --dev-node adapter name in the registry
* returning the GUID and optional actual_name.
*/
static const char *
get_device_guid (const char *name,
								 char *actual_name,
								 int actual_name_size,
								 const struct tap_reg *tap_reg,
								 const struct panel_reg *panel_reg,
struct gc_arena *gc)
{
	struct buffer ret = alloc_buf_gc (256, gc);
	struct buffer actual = clear_buf ();

	/* Make sure we have at least one TAP adapter */
	if (!tap_reg)
		return NULL;

	/* The actual_name output buffer may be NULL */
	if (actual_name)
	{
		ASSERT (actual_name_size > 0);
		buf_set_write (&actual, actual_name, actual_name_size);
	}

	/* Check if GUID was explicitly specified as --dev-node parameter */
	if (is_tap_win32 (name, tap_reg))
	{
		const char *act = guid_to_name (name, panel_reg);
		buf_printf (&ret, "%s", name);
		if (act)
			buf_printf (&actual, "%s", act);
		else
			buf_printf (&actual, "NULL");
		return BSTR (&ret);
	}

	/* Lookup TAP adapter in network connections list */
	{
		const char *guid = name_to_guid (name, tap_reg, panel_reg);
		if (guid)
		{
			buf_printf (&actual, "%s", name);
			buf_printf (&ret, "%s", guid);
			return BSTR (&ret);
		}
	}

	return NULL;
}
#endif	//0

bool EXIDevice_BBA::activate() {
	if(isActivated())
		return true;

	BBADEGUB("\nActivating BBA...\n");

	DWORD len;

#if 0
	device_guid = get_device_guid (dev_node, guid_buffer, sizeof (guid_buffer), tap_reg, panel_reg, &gc);

	if (!device_guid)
		DEGUB("TAP-Win32 adapter '%s' not found\n", dev_node);

	/* Open Windows TAP-Win32 adapter */
	openvpn_snprintf (device_path, sizeof(device_path), "%s%s%s",
		USERMODEDEVICEDIR,
		device_guid,
		TAPSUFFIX);
#endif	//0

	mHAdapter = CreateFile (/*device_path*/
		USERMODEDEVICEDIR "{1B1F9D70-50B7-4F45-AA4A-ABD17451E736}" TAPSUFFIX,
		GENERIC_READ | GENERIC_WRITE, 0, 0, OPEN_EXISTING,
		FILE_ATTRIBUTE_SYSTEM | FILE_FLAG_OVERLAPPED, 0);
	BBADEGUB("TAP-WIN32 device opened: %s\n",
		USERMODEDEVICEDIR "{1B1F9D70-50B7-4F45-AA4A-ABD17451E736}" TAPSUFFIX);
	if(mHAdapter == INVALID_HANDLE_VALUE) {
		FAIL(UE_BBA_ERROR);
	}

	/* get driver version info */
	{
		ULONG info[3];
		ZERO_ARRAY(info);
		if (DeviceIoControl (mHAdapter, TAP_IOCTL_GET_VERSION,
			&info, sizeof (info), &info, sizeof (info), &len, NULL))
		{
			BBADEGUB("TAP-Win32 Driver Version %d.%d %s\n",
				info[0], info[1], (info[2] ? "(DEBUG)" : ""));
		}
		if ( !(info[0] > TAP_WIN32_MIN_MAJOR
			|| (info[0] == TAP_WIN32_MIN_MAJOR && info[1] >= TAP_WIN32_MIN_MINOR)) )
		{
#define PACKAGE_NAME "WhineCube"
			DEGUB("ERROR:  This version of " PACKAGE_NAME " requires a TAP-Win32 driver that is at least version %d.%d -- If you recently upgraded your " PACKAGE_NAME " distribution, a reboot is probably required at this point to get Windows to see the new driver.\n",
				TAP_WIN32_MIN_MAJOR,
				TAP_WIN32_MIN_MINOR);
			FAIL(UE_BBA_ERROR);
		}
	}

	/* get driver MTU */
	{
		if(!DeviceIoControl(mHAdapter, TAP_IOCTL_GET_MTU,
			&mMtu, sizeof (mMtu), &mMtu, sizeof (mMtu), &len, NULL))
		{
			FAIL(UE_BBA_ERROR);
		}
		BBADEGUB("TAP-Win32 MTU=%d (ignored)\n", mMtu);
	}

	/* set driver media status to 'connected' */
	{
		ULONG status = TRUE;
		if (!DeviceIoControl (mHAdapter, TAP_IOCTL_SET_MEDIA_STATUS,
			&status, sizeof (status), &status, sizeof (status), &len, NULL))
		{
			DEGUB("WARNING: The TAP-Win32 driver rejected a TAP_IOCTL_SET_MEDIA_STATUS DeviceIoControl call.\n");
			FAIL(UE_BBA_ERROR);
		}
	}

	//set up recv event
	TGLE(mHRecvEvent = CreateEvent(NULL, false, false, NULL));
	ZERO_OBJECT(mReadOverlapped);
	TGLE(resume());

	BBADEGUB("Success!\n\n");
	return true;
}

bool EXIDevice_BBA::resume() {
	if(!isActivated())
		return true;
	BBADEGUB("BBA resume\n");
	//mStop = false;
	TGLE(RegisterWaitForSingleObject(&mHReadWait, mHRecvEvent, ReadWaitCallback,
		this, INFINITE, WT_EXECUTEDEFAULT));//WT_EXECUTEINWAITTHREAD
	mReadOverlapped.hEvent = mHRecvEvent;
	if(mBbaMem[BBA_NCRA] & BBA_NCRA_SR) {
		TGLE(startRecv());
	}
	BBADEGUB("BBA resume complete\n");
	return true;
}
bool EXIDevice_BBA::pause() {
	BBADEGUB("BBA pause\n");
	//mStop = true;
	//wait for callbacks to finish
	if(mHReadWait != INVALID_HANDLE_VALUE) {
		TGLE(UnregisterWaitEx(mHReadWait, INVALID_HANDLE_VALUE));
		mHReadWait = INVALID_HANDLE_VALUE;
	}
	BBADEGUB("BBA pause complete\n");
	return true;
}

bool EXIDevice_BBA::deactivate() {
	BBADEGUB("Deactivating BBA...\n");
	MYASSERT(isActivated());
	TGLE(CloseHandle(mHRecvEvent));
	mHRecvEvent = INVALID_HANDLE_VALUE;
	TGLE(CloseHandle(mHAdapter));
	mHAdapter = INVALID_HANDLE_VALUE;
	BBADEGUB("Success!\n");
	return true;
}

CriticalSectionWithSpin css_bba_recv;

bool EXIDevice_BBA::startRecv() {
	CSSHandler csshandler(css_bba_recv);
	MYASSERT(isActivated());
	BBADEGUB("startRecv... ");
	if(mWaiting) {
		BBADEGUB("already waiting\n");
		return true;
	}
	DWORD res = ReadFile(mHAdapter, mRecvBuffer, mRecvBuffer.size(),
		&mRecvBufferLength, &mReadOverlapped);
	if(res) {	//Operation completed immediately
		BBADEGUB("completed, res %i\n", res);
		mWaiting = true;
	} else {
		res = GetLastError();
		if (res == ERROR_IO_PENDING) {	//'s ok :)
			BBADEGUB("pending\n");
			//WaitCallback will be called
			mWaiting = true;
		}	else {	//error occurred
			FAIL(res);
		}
	}
	return true;
}

VOID CALLBACK EXIDevice_BBA::ReadWaitCallback(PVOID lpParameter, BOOLEAN TimerFired) {
	CSSHandler csshandler(css_bba_recv);
	try {
		static int sNumber = 0;
		int cNumber = sNumber++;
		BBADEGUB("WaitCallback %i\n", cNumber);
		MYASSERT(!TimerFired);
		EXIDevice_BBA* self = (EXIDevice_BBA*)lpParameter;
		MYASSERT(self->mHAdapter != INVALID_HANDLE_VALUE);
		HWGLE(GetOverlappedResult(self->mHAdapter, &self->mReadOverlapped,
			&self->mRecvBufferLength, false));
		self->mWaiting = false;
		HWGLE(self->handleRecvdPacket());
		BBADEGUB("WaitCallback %i done\n", cNumber);
	} catch(std::exception& e) {
		handleException(e);
	}
}

#if 0

void
close_tun (struct tuntap *tt)
{
	struct gc_arena gc = gc_new ();

	if (tt)
	{
#if 1
		if (tt->ipapi_context_defined)
		{
			DWORD status;
			if ((status = DeleteIPAddress (tt->ipapi_context)) != NO_ERROR)
			{
				DEGUB("Warning: DeleteIPAddress[%u] failed on TAP-Win32 adapter, status=%u : %s\n",
					(unsigned int)tt->ipapi_context,
					(unsigned int)status,
					strerror_win32 (status, &gc));
			}
		}
#endif

		if (tt->options.dhcp_release)
			dhcp_release (tt);

		if (tt->hand != NULL)
		{
			BBADEGUB("Attempting CancelIO on TAP-Win32 adapter\n");
			if (!CancelIo (tt->hand))
				DEGUB("Warning: CancelIO failed on TAP-Win32 adapter\n");
		}

		BBADEGUB("Attempting close of overlapped read event on TAP-Win32 adapter\n");
		overlapped_io_close (&tt->reads);

		BBADEGUB("Attempting close of overlapped write event on TAP-Win32 adapter\n");
		overlapped_io_close (&tt->writes);

		if (tt->hand != NULL)
		{
			BBADEGUB("Attempting CloseHandle on TAP-Win32 adapter\n");
			if (!CloseHandle (tt->hand))
				DEGUB("Warning: CloseHandle failed on TAP-Win32 adapter\n");
		}

		if (tt->actual_name)
			free (tt->actual_name);

		clear_tuntap (tt);
		free (tt);
	}
	gc_free (&gc);
}

int
tun_read_queue (struct tuntap *tt, int maxsize)
{
	if (tt->reads.iostate == IOSTATE_INITIAL)
	{
		DWORD len;
		BOOL status;
		int err;

		/* reset buf to its initial state */
		tt->reads.buf = tt->reads.buf_init;

		len = maxsize ? maxsize : BLEN (&tt->reads.buf);
		ASSERT (len <= BLEN (&tt->reads.buf));

		/* the overlapped read will signal this event on I/O completion */
		ASSERT (ResetEvent (tt->reads.overlapped.hEvent));

		status = ReadFile(
			tt->hand,
			BPTR (&tt->reads.buf),
			len,
			&tt->reads.size,
			&tt->reads.overlapped
			);

		if (status) /* operation completed immediately? */
		{
			/* since we got an immediate return, we must signal the event object ourselves */
			ASSERT (SetEvent (tt->reads.overlapped.hEvent));

			tt->reads.iostate = IOSTATE_IMMEDIATE_RETURN;
			tt->reads.status = 0;

			dmsg (D_WIN32_IO, "WIN32 I/O: TAP Read immediate return [%d,%d]\n",
				(int) len,
				(int) tt->reads.size);	       
		}
		else
		{
			err = GetLastError (); 
			if (err == ERROR_IO_PENDING) /* operation queued? */
			{
				tt->reads.iostate = IOSTATE_QUEUED;
				tt->reads.status = err;
				dmsg (D_WIN32_IO, "WIN32 I/O: TAP Read queued [%d]\n",
					(int) len);
			}
			else /* error occurred */
			{
				struct gc_arena gc = gc_new ();
				ASSERT (SetEvent (tt->reads.overlapped.hEvent));
				tt->reads.iostate = IOSTATE_IMMEDIATE_RETURN;
				tt->reads.status = err;
				dmsg (D_WIN32_IO, "WIN32 I/O: TAP Read error [%d] : %s\n",
					(int) len,
					strerror_win32 (status, &gc));
				gc_free (&gc);
			}
		}
	}
	return tt->reads.iostate;
}

int
tun_write_queue (struct tuntap *tt, struct buffer *buf)
{
	if (tt->writes.iostate == IOSTATE_INITIAL)
	{
		BOOL status;
		int err;

		/* make a private copy of buf */
		tt->writes.buf = tt->writes.buf_init;
		tt->writes.buf.len = 0;
		ASSERT (buf_copy (&tt->writes.buf, buf));

		/* the overlapped write will signal this event on I/O completion */
		ASSERT (ResetEvent (tt->writes.overlapped.hEvent));

		status = WriteFile(
			tt->hand,
			BPTR (&tt->writes.buf),
			BLEN (&tt->writes.buf),
			&tt->writes.size,
			&tt->writes.overlapped
			);

		if (status) /* operation completed immediately? */
		{
			tt->writes.iostate = IOSTATE_IMMEDIATE_RETURN;

			/* since we got an immediate return, we must signal the event object ourselves */
			ASSERT (SetEvent (tt->writes.overlapped.hEvent));

			tt->writes.status = 0;

			dmsg (D_WIN32_IO, "WIN32 I/O: TAP Write immediate return [%d,%d]\n",
				BLEN (&tt->writes.buf),
				(int) tt->writes.size);	       
		}
		else
		{
			err = GetLastError (); 
			if (err == ERROR_IO_PENDING) /* operation queued? */
			{
				tt->writes.iostate = IOSTATE_QUEUED;
				tt->writes.status = err;
				dmsg (D_WIN32_IO, "WIN32 I/O: TAP Write queued [%d]\n",
					BLEN (&tt->writes.buf));
			}
			else /* error occurred */
			{
				struct gc_arena gc = gc_new ();
				ASSERT (SetEvent (tt->writes.overlapped.hEvent));
				tt->writes.iostate = IOSTATE_IMMEDIATE_RETURN;
				tt->writes.status = err;
				dmsg (D_WIN32_IO, "WIN32 I/O: TAP Write error [%d] : %s\n",
					BLEN (&tt->writes.buf),
					strerror_win32 (err, &gc));
				gc_free (&gc);
			}
		}
	}
	return tt->writes.iostate;
}

int
tun_finalize (
							HANDLE h,
struct overlapped_io *io,
struct buffer *buf)
{
	int ret = -1;
	BOOL status;

	switch (io->iostate)
	{
	case IOSTATE_QUEUED:
		status = GetOverlappedResult(
			h,
			&io->overlapped,
			&io->size,
			FALSE
			);
		if (status)
		{
			/* successful return for a queued operation */
			if (buf)
				*buf = io->buf;
			ret = io->size;
			io->iostate = IOSTATE_INITIAL;
			ASSERT (ResetEvent (io->overlapped.hEvent));
			dmsg (D_WIN32_IO, "WIN32 I/O: TAP Completion success [%d]\n", ret);
		}
		else
		{
			/* error during a queued operation */
			ret = -1;
			if (GetLastError() != ERROR_IO_INCOMPLETE)
			{
				/* if no error (i.e. just not finished yet),
				then DON'T execute this code */
				io->iostate = IOSTATE_INITIAL;
				ASSERT (ResetEvent (io->overlapped.hEvent));
				msg (D_WIN32_IO | M_ERRNO, "WIN32 I/O: TAP Completion error\n");
			}
		}
		break;

	case IOSTATE_IMMEDIATE_RETURN:
		io->iostate = IOSTATE_INITIAL;
		ASSERT (ResetEvent (io->overlapped.hEvent));
		if (io->status)
		{
			/* error return for a non-queued operation */
			SetLastError (io->status);
			ret = -1;
			msg (D_WIN32_IO | M_ERRNO, "WIN32 I/O: TAP Completion non-queued error\n");
		}
		else
		{
			/* successful return for a non-queued operation */
			if (buf)
				*buf = io->buf;
			ret = io->size;
			dmsg (D_WIN32_IO, "WIN32 I/O: TAP Completion non-queued success [%d]\n", ret);
		}
		break;

	case IOSTATE_INITIAL: /* were we called without proper queueing? */
		SetLastError (ERROR_INVALID_FUNCTION);
		ret = -1;
		dmsg (D_WIN32_IO, "WIN32 I/O: TAP Completion BAD STATE\n");
		break;

	default:
		ASSERT (0);
	}

	if (buf)
		buf->len = ret;
	return ret;
}

#endif	//0
